import {
  ElMessage
} from 'element-plus'

import {
  getSnowflake,
  uuid,
  pointToGrid,
  equals,
  dist2d,
  angle,
  getNearestCoord,
  pointToFootDist,
  pointInSegments,
  filterPointInSegments,
  getSlope,
  collinear3Points,
  intersects,
  lineIntersect
} from './algorithm'

import nodesDefault, { gutter } from './nodesDefault'
import type { INodes, Nodes } from './INodes'

export {
  getSnowflake,
  uuid,
  pointToGrid,
  dist2d,
  angle,
  getNearestCoord,
  pointToFootDist,
  pointInSegments,
  filterPointInSegments,
  getSlope,
  collinear3Points,
  intersects,
  lineIntersect
} from './algorithm' // 默认节点属性对象

/**
 * 创建新节点
 * @author 谁改填谁
 * @param {Object} anonymous node节点对象属性
 * @returns {Object} node节点对象
 */
export function createNode({
  id = getSnowflake(), // 节点唯一标识id
  nodeType = '0',
  nodeCode = uuid(), // 用户可编辑节点code
  nodeName = '流程开始', // 节点名称
  paths = [],
  position = {
    x: 300,
    y: 20
  },
  size = {
    width: 40,
    height: 40
  }
} = {}, params = {}) {
  return {
    'id': id,
    'nodeCode': nodeCode,
    'nodeType': nodeType,
    'nodeName': nodeName,
    'nodePath': {
      position: position,
      size: size,
      /**
       * 开始节点, 结束节点 nodePath
       * demo 实际数据自己修改
       */
      // paths: createPath({
      //   position: {x: 292, y: 20},
      //   size: {width: 32, height: 32}
      // }, {
      //   position: {x: 360, y: 20},
      //   size: {width: 32, height: 32}
      // })
      paths: paths
    },
    ...params
  }
}

/**
 * 获取节点基础属性
 * @param {Object} node
 * @returns {Object} x,y,width,height,top,bottom,left,right
 */
export function getNodeBasic(node) {
  const { x, y } = node.nodePath.position
  const width = node.nodePath.size.width
  const height = node.nodePath.size.height
  return {
    x,
    y,
    width,
    height,
    top: y - height / 2,
    bottom: y + height / 2,
    left: x - width / 2,
    right: x + width / 2
  }
}

/**
 * 判断点是否在矩形内
 * @param {Array} point [100,100]
 * @param {Object} node
 * @returns {Boolean}
 */
export function inRect(point, node) {
  const { 0: x, 1: y } = point
  const { position, size } = node.nodePath
  let { x: x1, y: y1 } = position
  x1 = x1 - size.width / 2
  y1 = y1 - size.height / 2
  const x2 = x1 + node.nodePath.size.width
  const y2 = y1 + node.nodePath.size.height
  return x >= x1 && x <= x2 && y >= y1 && y <= y2
}
// 不包含边框
export function insideRect(point, node) {
  const { 0: x, 1: y } = point
  const { position, size } = node.nodePath
  let { x: x1, y: y1 } = position
  x1 = x1 - size.width / 2
  y1 = y1 - size.height / 2
  const x2 = x1 + node.nodePath.size.width
  const y2 = y1 + node.nodePath.size.height
  return x > x1 && x < x2 && y > y1 && y < y2
}
// 增加额外吸附距离
export function inRectAdsorb(point, node) {
  const adsorb = 10 * 2
  const { 0: x, 1: y } = point
  const { position, size } = node.nodePath
  let { x: x1, y: y1 } = position
  const halfWidth = (size.width + adsorb) / 2
  const halfHeight = (size.height + adsorb) / 2
  const x2 = x1 + halfWidth
  const y2 = y1 + halfHeight
  x1 = x1 - halfWidth
  y1 = y1 - halfHeight
  return x >= x1 && x <= x2 && y >= y1 && y <= y2
}

// 2个点是否在水平线或垂直线上
export function isVertical(p1, p2) {
  const { 0: x1, 1: y1 } = p1
  const { 0: x2, 1: y2 } = p2
  return x1 === x2 || y1 === y2
}

/**
 * actives多选，选中状态处理
 * 主要用在组合按键，Shift+鼠标点击
 * @param {Array} actives actives原型数组
 * @param {String} id 点击节点id
 */
export function checkActives(actives, id) {
  const index = actives.indexOf(id)
  if (index > -1) {
    actives.splice(index, 1)
  } else {
    actives.push(id)
  }
}

/**
 * 计算画布中坐标
 * @param {Number} pageX e.pageX
 * @param {Number} pageY e.pageX
 * @param {Number} oldPageX mousedown的pageX
 * @param {Number} oldPageY mousedown的pageY
 * @param {Number} x 画布的水平偏移
 * @param {Number} y 画布的垂直偏移
 * @param {Number} oldX mousedown的this.x
 * @param {Number} oldY mousedown的this.y
 * @param {Number} scale this.scale
 * @param {Number} oldScale mousedown时的this.scale
 * @param {Number} diffX 补x
 * @param {Number} diffY 补y
 */
export function getMovingPoint({
  pageX = 0,
  pageY = 0,
  oldPageX = 0,
  oldPageY = 0,
  x = 0,
  y = 0,
  oldX = 0,
  oldY = 0,
  scale = 1,
  oldScale = 1,
  diffX = 0,
  diffY = 0
} = {}) {
  return [
    pointToGrid((pageX - oldPageX - (x - oldX)) / scale + diffX),
    pointToGrid((pageY - oldPageY - (y - oldY)) / scale + diffY)
  ]
}

/**
 * 计算默认节点在画布中的中心坐标
 * @param {Array} nodeConfig
 * @param {Number} x
 * @param {Number} y
 * @returns {Object} {x,y}
 */
export function getCenterCoords(nodeConfig, svgWrap, x, y) {
  if (x !== 0) {
    x = 0
  }
  if (y !== 0) {
    y = 0
  }
  let startNodesX = nodeConfig.find(item => item.nodeType === '1').nodePath.position.x
  let startNodesY = nodeConfig.find(item => item.nodeType === '1').nodePath.position.y
  let pageOffsetWidth = svgWrap.width.baseVal.value
  return {
    x: x + 0.5 * pageOffsetWidth - startNodesX,
    y: y + 80 - startNodesY
  }
}


/**
 * 画布相对所有节点中心点
 * @param {Array} nodeConfig 数据
 * @returns {Object} center {scale,x,y}
 */
export function getCenter(nodeConfig, svgWrap) {
  const pageOffsetWidth = svgWrap.width.baseVal.value
  const pageOffsetHeight = svgWrap.height.baseVal.value
  const coordsX: number[] = []
  const coordsY: number[] = []
  // 折叠节点的嵌套子节点 / 顶层以外节点
  // let nodes = JSON.parse(JSON.stringify(nodeConfig)).filter(node => !node.foldNest && !node.exceptTop)
  let nodes = nodeConfig.filter(node => !node.foldNest && !node.exceptTop)
  nodes.forEach(item => {
    const position = item.position
    const size = item.size
    coordsX.push(position.x - size.width / 2)
    coordsX.push(position.x + size.width / 2)
    coordsY.push(position.y - size.height / 2)
    coordsY.push(position.y + size.height / 2)
  })
  const minX = Math.min.apply(null, coordsX)
  const minY = Math.min.apply(null, coordsY)
  const maxX = Math.max.apply(null, coordsX)
  const maxY = Math.max.apply(null, coordsY)
  const width = maxX - minX
  const height = maxY - minY
  const center = {
    x: maxX - width / 2,
    y: maxY - height / 2
  }
  const scale = Math.min((pageOffsetWidth - 80) / width, (pageOffsetHeight - 80) / height) // 各边留20px
  if (scale > 1) {
    let startNodesY = nodes[0].position.y
    return {
      scale: 1,
      x: 0.5 * pageOffsetWidth - center.x,
      y: 120 - startNodesY
    }
  }
  return {
    scale,
    // x: pageOffsetWidth / 2 + (width / 2 - center.x) * scale - width / 2 * scale,
    // y: pageOffsetHeight / 2 + (height / 2 - center.y) * scale - height / 2 * scale
    x: (pageOffsetWidth) / 2 - (minX + width / 2) * scale,
    y: (pageOffsetHeight) / 2 - (minY + height / 2) * scale
  }
}

// export function getCenter(nodes, svgWrap) {
//   const pageOffsetWidth = svgWrap.width.baseVal.value
//   const pageOffsetHeight = svgWrap.height.baseVal.value
//   const { width, height } = nodes.reduce((a, b) => {
//     console.log(b.size.boxHeight)
//     return {
//       width: a.width + b.size.boxWidth,
//       height: a.height + b.size.boxHeight
//     }
//   }, { width: 0, height: 0 })
//   console.log(width, pageOffsetWidth)
//   console.log(height, pageOffsetHeight)

//   const minX = nodes[0].position.left
//   const minY = nodes[0].position.top
//   // const maxX = nodes[0].position.left + width
//   // const maxY = nodes[0].position.top + height
//   const center = {
//     x: minX - width / 2,
//     y: minY - height / 2
//   }
//   const scale = Math.min(pageOffsetWidth / width, pageOffsetHeight / height) // 各边留20px
//   if (scale > 1) {
//     let startNodesY = nodes[0].position.top
//     return {
//       scale: 1,
//       x: 0.5 * pageOffsetWidth - center.x,
//       y: 80 - startNodesY
//     }
//   }
//   return {
//     scale,
//     x: (pageOffsetWidth) / 2 - (minX + width / 2) * scale,
//     y: (pageOffsetHeight) / 2 - (minY + height / 2) * scale
//   }
// }


/**
 * 生成 node 节点的每个坐标
 * @param {Array} treeNodes 树形节点数据
 * @param {Array} parent 树形节点数据
 * @returns {Nodes[]}
 */
export function getNodeConfigCoords(treeNodes: Nodes[], parent: Nodes | null = null, level = 0, heights: number[] = [], levelNodes: Nodes[][] = [[]]) {
  // 记录各个 level 的节点 最大高度
  heights[level] = treeNodes
    .filter(node => !['2'].includes(node.nodeType) && !node.lowerLevelNum && !node.fold)
    .reduce((a, b) => Math.max(a, nodesDefault[b.nodeType].size.height), nodesDefault[treeNodes[0].nodeType].size.height)
  return treeNodes.map((node, index, nodes) => {
    const lowerLevelNum = Number(node.lowerLevelNum ?? 0)
    const temLevel = level + lowerLevelNum
    node.level = parent?.level ? parent.level + 1 + lowerLevelNum : temLevel // 层级
    if (!levelNodes[node.level]) {
      levelNodes[node.level] = []
    }
    if (String(node.nodeType) !== '5') {
      levelNodes[node.level].push(node)
    }
    // node.fold = node.fold ?? false // 折叠状态
    const nodeDefault = nodesDefault[node.nodeType] // 默认节点数据
    // 初始大小
    node.size = {
      width: nodeDefault.size.width,
      height: nodeDefault.size.height
    }
    // -----------
    // 初始定位 开始
    // -----------
    // node.position = {
    //   left: 0,
    //   top: node.size.boxHeight * node.level,
    //   x: node.size.boxWidth / 2,
    //   y: node.size.boxHeight * node.level + node.size.height / 2
    // }
    node.position = {}
    // node.position.left = 0
    node.position.top = (parent?.position?.top ?? 0) + (parent?.size?.height ?? 0) + gutter.y * 2
    // 处理下增节点 top
    if (lowerLevelNum) {
      const heightsSum = heights.slice(level, temLevel).reduce((a, b) => a + b, 0)
      const lowerGutterSum = lowerLevelNum * gutter.y * 2
      node.position.top += heightsSum + lowerGutterSum
    }
    node.position.y = node.position.top + node.size.height / 2
    if (parent) {
      /**
       * 取当前节点,左侧同级节点x
       * 过滤助理节点
       */
      const maxX = Math.max(...levelNodes[node.level].map(item => item.position?.x || -Infinity))
      if (index === 0) {
        // 当前树当前层级，首个节点
        const filterNodes = nodes.filter(p => String(p.nodeType) !== '5')
        let filterWidth = filterNodes.reduce((a, b, index, arr) => {
          if (arr.length === 1) return a
          if (index === 0) {
            return a + nodesDefault[b.nodeType].size.width / 2 + gutter.x
          } else if (index === arr.length - 1) {
            return a + nodesDefault[b.nodeType].size.width / 2
          }
          return a + nodesDefault[b.nodeType].size.width + gutter.x
        }, 0)
        const x = parent.position.x - (filterWidth / 2)
        if (x - node.size.width - gutter.x > maxX) {
          // 首个节点大于左侧同级节点x
          node.position.x = x
        } else {
          // 首个右侧节点x
          node.position.x = maxX + node.size.width + gutter.x
        }
      } else if (String(node.nodeType) === '5') {
        // 忽略助理节点
      } else {
        // 正常节点
        node.position.x = maxX + node.size.width + gutter.x
      }
    } else {
      // 顶层节点
      node.position.x = 0
    }
    // -----------
    // 初始定位 结束
    // -----------
    const hasChildren = Array.isArray(node.children) && !!node.children.length
    // 开始处理子节点
    if (hasChildren && !node.fold) {
      // ----
      // 递归
      // ----
      node.children = getNodeConfigCoords(node.children!, node, node.level + 1, heights, levelNodes)
    }
    // 清除末级节点 children
    if (Array.isArray(node.children) && !node.children.length || node.fold) {
      delete node.paths
    }
    // 父节点 向下处理子节点的 特殊属性和连线
    if (hasChildren && !node.fold) {
      node.children?.forEach((child, index, childrens) => {
        if (String(child.nodeType) === '5') {
          // 助理
          child.position.x = node.position.x + child.size.width / 2 + node.size.width * 0.1
          child.position.y = node.position.top + node.size.height + gutter.y * 0.75
          child.position.top = child.position.y - child.size.height / 2
        } else {
          // 正常节点
          node.position.x = childrens[0].position.x + (childrens[childrens.length - 1].position.x - childrens[0].position.x) / 2
        }
      })
      // 画线
      node.paths = node.children?.map(child => {
        // 助理
        const isAssistant = String(child.nodeType) === '5'
        if (isAssistant) {
          return {
            points: [
              [node.position.x, node.position.top + node.size.height],
              [node.position.x, (node.position.top + node.size.height) + gutter.y * 0.75],
              [node.position.x + node.size.width * 0.1 + gutter.x, (node.position.top + node.size.height) + gutter.y * 0.75]
            ]
          }
        }
        return {
          points: [
            [node.position.x, node.position.top + node.size.height],
            [node.position.x, (node.position.top + node.size.height) + gutter.y * 1.5],
            [child.position.x, (node.position.top + node.size.height) + gutter.y * 1.5],
            [child.position.x, child.position.top]
          ]
        }
      })
    }
    return node
  })
}

/**
 * 生成紧凑型树状图算法
 */
export function generateCompactTreeAlgorithm() {
  return {
    getNodeConfigCoords
  }
}